<?php
/**
 * Plugin Name: LMN Site Monitor
 * Plugin URI:  https://lumina.fmkr.net/site-monitor
 * Description: Monitor de sitios desde el panel de administración de WordPress, con pings HTTP/HTTPS, caducidad SSL y alertas por email.
 * Version:     0.6.4
 * Author:      Francisco Moreno
 * Author URI:  https://fcomoreno.net
 * License:     GPL-2.0-or-later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: lmn-site-monitor
 * Domain Path: /languages
 *
 * @package LMN_Site_Monitor
 * @author Francisco Moreno
 * @license GPLv3
 * @since 0.6.4 Sanitización completa y 0 warnings
 */

if ( !defined( 'ABSPATH' ) )exit;

class LM_Site_Monitor_Classic {
  private $option_sites = 'lmn_sites';
  private $option_settings = 'lmn_settings';
  private $option_last_alerts = 'lmn_last_alerts';
  const CRON_HOOK = 'lmn_check_sites_cron';
  const MENU_SLUG = 'lmn-site-monitor';
  const VERSION = '0.6.4';

  public function __construct() {
    // Admin UI
    add_action( 'admin_menu', [ $this, 'admin_menu' ] );
    add_action( 'admin_enqueue_scripts', [ $this, 'admin_assets' ] );

    // Form handlers (admin-post)
    add_action( 'admin_post_lm_add_site', [ $this, 'handle_add_site' ] );
    add_action( 'admin_post_lm_delete_site', [ $this, 'handle_delete_site' ] );
    add_action( 'admin_post_lm_force_check', [ $this, 'handle_force_check' ] );
    add_action( 'admin_post_lm_save_settings', [ $this, 'handle_save_settings' ] );
    add_action( 'admin_post_lm_check_all', [ $this, 'handle_check_all' ] );
    add_action( 'admin_post_lm_save_widget_prefs', [ $this, 'handle_save_widget_prefs' ] );
    add_action( 'admin_post_lm_export_csv', [ $this, 'handle_export_csv' ] ); // NUEVO
    add_action( 'admin_post_lm_import_csv', [ $this, 'handle_import_csv' ] ); // NUEVO
    add_action( 'admin_post_lm_bulk_action', [ $this, 'handle_bulk_action' ] ); // NUEVO

    // Settings & Cron
    add_action( 'admin_init', [ $this, 'maybe_boot_defaults' ] );
    add_filter( 'cron_schedules', [ $this, 'add_cron_intervals' ] );
    add_action( self::CRON_HOOK, [ $this, 'check_all_sites_cron' ] );
    add_action( 'phpmailer_init', [ $this, 'apply_smtp' ] );
    add_action( 'wp_mail_succeeded', [ $this, 'handle_wp_mail_succeeded' ] );
    add_action( 'wp_mail_failed', [ $this, 'handle_wp_mail_failed' ] );
    add_action( 'wp_mail_failed', [ $this, 'on_mail_failed' ] );
    add_action( 'admin_notices', [ $this, 'admin_notice_mail_failed' ] );
    register_activation_hook( __FILE__, [ $this, 'activate' ] );
    register_deactivation_hook( __FILE__, [ $this, 'deactivate' ] );

    // Dashboard widget
    add_action( 'wp_dashboard_setup', [ $this, 'dashboard_widget' ] );
  }

  /* ---------- Activation ---------- */
  public function activate() {
    $this->maybe_boot_defaults();
    $settings = get_option( $this->option_settings );
    if ( !wp_next_scheduled( self::CRON_HOOK ) ) {
      wp_schedule_event( time() + 60, $settings[ 'interval' ], self::CRON_HOOK );
    }
  }
  public function deactivate() {
    wp_clear_scheduled_hook( self::CRON_HOOK );
  }

  public function maybe_boot_defaults() {
    $defaults = [
      'admin_email' => get_option( 'admin_email' ),
      'interval' => 'fivemin', // fivemin, fifteenmin, hourly, daily
      'days_before_expiry' => 5,
      'throttle_hours' => 12,
      'default_theme' => 'system', // light | dark | system
      'use_local_font' => 0, // desactivado por defecto
      'allow_insecure_pings' => 0, // pings con sslverify=false (opcional)
    ];
    $cur = get_option( $this->option_settings );
    if ( !$cur )add_option( $this->option_settings, $defaults );
    else update_option( $this->option_settings, array_merge( $defaults, $cur ) );
  }

  /* ---------- Assets (CSS/JS) ---------- */
  public function admin_assets( $hook ) {
    // Cargar CSS/JS en nuestra pantalla y en el Escritorio (para el widget)
    $is_our_page = ( $hook === 'toplevel_page_' . self::MENU_SLUG );
    if ( $is_our_page || $hook === 'index.php' ) {
      $ver = self::VERSION;
      wp_enqueue_style( 'lm-monitor-theme', plugins_url( 'assets/admin-theme.css', __FILE__ ), [], $ver );

      $settings = get_option( $this->option_settings, [] );
      wp_enqueue_script( 'lm-monitor-js', plugins_url( 'assets/admin-theme.js', __FILE__ ), [], $ver, true );
      wp_add_inline_script( 'lm-monitor-js', 'window.LM_MONITOR_SETTINGS = ' . wp_json_encode( [
        'defaultTheme' => $settings[ 'default_theme' ] ?? 'system',
        'useLocalFont' => ( int )( $settings[ 'use_local_font' ] ?? 0 ),
        'fontsBaseUrl' => plugins_url( 'assets/fonts/', __FILE__ ),
      ] ) . ';' );
    }
  }

  /* ---------- Cron intervals ---------- */
  public function add_cron_intervals( $schedules ) {
    $schedules[ 'fivemin' ] = [ 'interval' => 300, 'display' => __( 'Every 5 Minutes', 'lmn-site-monitor' ) ];
    $schedules[ 'fifteenmin' ] = [ 'interval' => 900, 'display' => __( 'Every 15 Minutes', 'lmn-site-monitor' ) ];
    $schedules[ 'hourly' ] = [ 'interval' => 3600, 'display' => __( 'Hourly', 'lmn-site-monitor' ) ];
    $schedules[ 'daily' ] = [ 'interval' => 86400, 'display' => __( 'Daily', 'lmn-site-monitor' ) ];
    return $schedules;
  }

  /* ---------- Admin UI ---------- */
  public function admin_menu() {
    add_menu_page( 'Site Monitor', 'Site Monitor', 'manage_options', self::MENU_SLUG, [ $this, 'admin_page' ], 'dashicons-networking', 60 );
  }

  private function header_toolbar() {
    // Toggle rápido de tema (persistente con localStorage)
    echo '<div class="lm-toolbar">
                <div class="lm-description postbox lm-card">LMN Site Monitor es un monitor de estado para sitios web. Monitoriza múltiples sitios web desde el panel de administración de WordPress: pings periódicos, chequeo de caducidad del certificado SSL, alertas por email, widget de Escritorio y exportación CSV.
                  <ul> 
                    <li>Añade las URL a monitorizar desde el campo "Añadir URL". Puedes hacer una copia de seguridad de tu listado de URLs exportando a CSV.</li> 
                    <li>Podrás importar tu archivo CSV en una nueva instalación del plugin. </li> 
                    <li>Recuerda configurar un SMTP válido si tu WordPress no lo tiene ya configurado. </li> 
                    <li>En la configuración del plugin podrás añadir un servicio SMTP externo para que se envíen las notificaciones.</li> 
                  </ul>
                 </div>
                <div class="lm-theme-toggle">
                    <button type="button" class="button lm-theme-btn" data-mode="light" title="Tema claro">☀️</button>
                    <button type="button" class="button lm-theme-btn" data-mode="dark" title="Tema oscuro">🌙</button>
                    <button type="button" class="button lm-theme-btn" data-mode="system" title="Seguir sistema">🖥️</button>
                </div>
              </div>';
  }

  public
  function admin_page() {
    if ( !current_user_can( 'manage_options' ) )wp_die( 'No autorizado' );
    $sites = get_option( $this->option_sites, [] );
    $settings = get_option( $this->option_settings, [] );
    $msg = isset($_GET['msg']) ? sanitize_text_field( wp_unslash( $_GET['msg'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- benign admin notice

    // Datos para "Acerca"
    $intervalMap = [
      'fivemin' => 'Cada 5 minutos',
      'fifteenmin' => 'Cada 15 minutos',
      'hourly' => 'Cada hora',
      'daily' => 'Diario'
    ];
    $intervalLbl = $intervalMap[ $settings[ 'interval' ] ?? 'fivemin' ] ?? ( $settings[ 'interval' ] ?? '—' );
    $nextRunTs = wp_next_scheduled( self::CRON_HOOK );
    $nextRun = $nextRunTs ? date_i18n( 'Y-m-d H:i:s', $nextRunTs ) : '—';
    $cronOff = ( defined( 'DISABLE_WP_CRON' ) && DISABLE_WP_CRON ) ? 'Sí' : 'No';

    $ok = 0;
    $down = 0;
    $expSoon = 0;
    $days_before = intval( $settings[ 'days_before_expiry' ] ?? 5 );
    foreach ( $sites as $s ) {
      $status = $s[ 'last_status' ] ?? '';
      if ( $status === 'down' )$down++;
      else $ok++;
      if ( !empty( $s[ 'ssl_expires' ] ) ) {
        $d = ceil( ( $s[ 'ssl_expires' ] - time() ) / 86400 );
        if ( $d <= $days_before )$expSoon++;
      }
    }

   // Inicializar traducciones del plugin.
    add_action( 'init', function() {
    }); 
    ?>
<div class="wrap lm-scope lmn-myheader">
  <h1>LMN Site Monitor <span class="lmn-version-badge">V.&nbsp;<?php echo esc_html(self::VERSION); ?></span></h1>
  <?php $this->header_toolbar(); ?>
  <?php if ($msg): ?>
  <div class="notice notice-success is-dismissible">
    <p><?php echo esc_html($msg); ?></p>
  </div>
  <?php endif; ?>
  
  <!-- Tabs (sin JS, con :target) -->
  <?php $tab = isset($_GET['tab']) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'sites'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- view state only ?>
<nav class="lm-tabs" aria-label="Secciones">
    <?php

    // Fallback a hash si llega con # en URL
    ?>
    <a href="<?php echo esc_url( add_query_arg('tab', 'sites') ); ?>" class="<?php echo ($tab==='sites')?'is-active':''; ?>">Sitios</a> <a href="<?php echo esc_url( add_query_arg('tab', 'config') ); ?>" class="<?php echo ($tab==='config')?'is-active':''; ?>">Configuración</a> <a href="<?php echo esc_url( add_query_arg('tab', 'changelog') ); ?>" class="<?php echo ($tab==='changelog')?'is-active':''; ?>">
    <?php esc_html_e('Changelog', 'lmn-site-monitor'); ?>
    </a> </nav>
  
  <!-- Panel: Sitios -->
  <section class="lm-tabpanel <?php echo ($tab==='sites')?'is-visible':''; ?>" id="tab-sites">
    <div class="lm-grid"> 
      
      <!-- Banner Lúmina (aside lateral derecho, sólo escritorio) -->
      <aside class="lm-banner-lumina" aria-label="<?php esc_attr_e('Lúmina – Verificador de accesibilidad','lmn-site-monitor'); ?>"> <a href="https://lumina.fmkr.net/" target="_blank" rel="noopener noreferrer"> <img
              src="<?php echo esc_url( plugins_url('assets/banner-lumina.webp', __FILE__) ); ?>"
              alt="<?php esc_attr_e('Comprueba que tu web sea accesible – WCAG 2.1/2.2 (Lúmina)', 'lmn-site-monitor'); ?>"
              title="<?php esc_attr_e('Comprueba que tu web sea accesible – WCAG 2.1/2.2 (Lúmina)', 'lmn-site-monitor'); ?>"
              loading="lazy"
              decoding="async"
            /> </a> </aside>
      <div class="metabox-holder lm-main">
        <div class="postbox lm-card">
          <h2 class="hndle"><span>Sitios monitorizados</span></h2>
          <div class="inside">
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" class="lm-row">
              <?php wp_nonce_field('lm_add_site','lm_add_site_nonce'); ?>
              <input type="hidden" name="action" value="lm_add_site">
              <p class="lm-grow">
                <label for="lm_url">Añadir URL</label>
                <input id="lm_url" name="url" type="url" class="regular-text" placeholder="https://ejemplo.com" required >
              </p>
              <?php submit_button('Añadir', 'primary', 'lm_add_submit', false, ['id'=>'lm-add']); ?>
            </form>
            <div class="lm-cards-wrap">
              <form id="lm-bulk-form" method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" class="lm-bulkbar">
                <?php wp_nonce_field('lm_bulk_action','lm_bulk_nonce'); ?>
                <input type="hidden" name="action" value="lm_bulk_action">
                <select name="bulk_action">
                  <option value="">
                  <?php esc_html_e('Acciones en lote','lmn-site-monitor'); ?>
                  </option>
                  <option value="delete">
                  <?php esc_html_e('Eliminar','lmn-site-monitor'); ?>
                  </option>
                </select>
                <?php submit_button(__('Aplicar','lmn-site-monitor'), 'secondary', 'lm_bulk_apply', false, ['id'=>'lm-bulk-apply']); ?>
              </form>
              <div class="lm-cards" role="list">
                <?php if (empty($sites)): ?>
                <div class="lm-card-item lm-empty" role="listitem">
                  <p>
                    <?php esc_html_e('No hay sitios configurados.','lmn-site-monitor'); ?>
                  </p>
                </div>
                <?php
                else :
                  uasort( $sites, function ( $a, $b ) {
                    $sa = $a[ 'last_status' ] ?? '';
                    $sb = $b[ 'last_status' ] ?? '';
                    if ( $sa === 'down' && $sb !== 'down' ) return -1;
                    if ( $sb === 'down' && $sa !== 'down' ) return 1;
                    return strcmp( $a[ 'url' ], $b[ 'url' ] );
                  } );
                foreach ( $sites as $id => $s ):
                  $checked = isset( $s[ 'last_checked' ] ) ? date_i18n( 'Y-m-d H:i:s', $s[ 'last_checked' ] ) : '—';
                $status = $s[ 'last_status' ] ?? '';
                $http = $s[ 'http_code' ] ?? '—';
                $sslD = isset( $s[ 'ssl_expires' ] ) ? ceil( ( $s[ 'ssl_expires' ] - time() ) / 86400 ) : null;
                $sslF = isset( $s[ 'ssl_expires' ] ) ? date_i18n( 'Y-m-d H:i', $s[ 'ssl_expires' ] ) : '—';
                $badge = ( $sslD !== null ) ? '<span class="lm-days ' . ( $sslD < 15 ? 'down' : ( $sslD < 30 ? 'warn' : '' ) ) . '">' . esc_html( $sslD . ' ' . __( 'días', 'lmn-site-monitor' ) ) . '</span>': '—';
                ?>
                <article class="lm-card-item" role="listitem" data-status="<?php echo esc_attr($status ?: 'unknown'); ?>">
                  <header class="lm-card-item__head">
                    <label class="lm-card-check">
                      <input class="lm-bulk-cb" type="checkbox" name="ids[]" form="lm-bulk-form" value="<?php echo esc_attr($id); ?>">
                    </label>
                    <div class="lm-card-url"> <a href="<?php echo esc_url($s['url']); ?>" target="_blank" rel="noreferrer"><?php echo esc_html($s['url']); ?></a> </div>
                    <div class="lm-card-state">
                      <?php if ($status === 'down'): ?>
                      <span class="lm-badge down">
                      <?php esc_html_e('Caído','lmn-site-monitor'); ?>
                      </span>
                      <?php elseif ($status === 'ok'): ?>
                      <span class="lm-badge ok">OK</span>
                      <?php else: ?>
                      —
                      <?php endif; ?>
                    </div>
                  </header>
                  <dl class="lm-card-meta">
                    <div>
                      <dt>
                        <?php esc_html_e('Última comprobación','lmn-site-monitor'); ?>
                      </dt>
                      <dd><?php echo esc_html($checked); ?></dd>
                    </div>
                    <div>
                      <dt>HTTP</dt>
                      <dd><?php echo esc_html($http); ?></dd>
                    </div>
                    <div>
                      <dt>
                        <?php esc_html_e('SSL caduca','lmn-site-monitor'); ?>
                      </dt>
                      <dd><?php echo esc_html($sslF); ?></dd>
                    </div>
                    <div>
                      <dt>
                        <?php esc_html_e('Días','lmn-site-monitor'); ?>
                      </dt>
                      <dd><?php echo wp_kses_post($badge); ?></dd>
                    </div>
                  </dl>
                  <footer class="lm-card-actions">
                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" class="lm-inline">
                      <?php wp_nonce_field('lm_force_check_'.$id, 'lm_force_nonce_'.$id); ?>
                      <input type="hidden" name="action" value="lm_force_check">
                      <input type="hidden" name="site_id" value="<?php echo esc_attr($id); ?>">
                      <?php submit_button(__('Forzar','lmn-site-monitor'), '', 'lm_force_' . $id, false, ['id'=>'lm-force-' . $id]); ?>
                    </form>
                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" class="lm-inline" onsubmit="return confirm('<?php echo esc_js(__('¿Eliminar sitio?','lmn-site-monitor')); ?>');">
                      <?php wp_nonce_field('lm_delete_site_'.$id, 'lm_delete_nonce_'.$id); ?>
                      <input type="hidden" name="action" value="lm_delete_site">
                      <input type="hidden" name="site_id" value="<?php echo esc_attr($id); ?>">
                      <?php submit_button(__('Eliminar','lmn-site-monitor'), 'secondary', 'lm_delete_' . $id, false, ['id'=>'lm-delete-' . $id]); ?>
                    </form>
                  </footer>
                </article>
                <?php endforeach; endif; ?>
              </div>
              <!-- .lm-cards --> 
            </div>
            <!-- .lm-cards-wrap -->
            
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin-top:12px;">
              <?php wp_nonce_field('lm_check_all','lm_check_all_nonce_a'); ?>
              <input type="hidden" name="action" value="lm_check_all">
              <?php submit_button('Comprobar todos ahora','primary','lm_check_all_submit_top',false,array( 'id' => 'lm-check-all-top' )); ?>
            </form>
            
            <!-- Import/Export -->
            <div class="lm-inline-actions" style="display:flex;flex-wrap:wrap;gap:8px;margin-top:8px;">
              <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                <?php wp_nonce_field('lm_export_csv','lm_export_nonce'); ?>
                <input type="hidden" name="action" value="lm_export_csv">
                <?php submit_button('Exportar CSV', 'secondary', 'lm_export_submit', false, ['id'=>'lm-export']); ?>
              </form>
              <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" enctype="multipart/form-data">
                <?php wp_nonce_field('lm_import_csv','lm_import_nonce'); ?>
                <input type="hidden" name="action" value="lm_import_csv">
                <input type="file" name="csv_file" accept=".csv" required>
                <?php submit_button('Importar CSV', 'secondary', 'lm_import_submit', false, ['id'=>'lm-import']); ?>
              </form>
            </div>
          </div>
        </div>
      </div>
    </div>
    <style>
          /* Banner Lúmina – oculto en móvil/tablet, visible en escritorio */
          .lm-banner-lumina { display: none; }
          @media (min-width: 1025px) {
            .lm-grid { position: relative; } /* referencia para posicionamiento absoluto */
            .lm-banner-lumina {
              display: block;
              position: absolute;
              right: 30px;
              top: 10px;
              width: 260px;
              z-index: 5;
            }
            .lm-banner-lumina img {
              display: block;
              width: 100%;
              height: auto;
              border-radius: 12px;
              box-shadow: var(--lm-shadow, 0 8px 24px rgba(0,0,0,.10));
            }
          }
        </style>
  </section>
  
  <!-- Panel: Configuración -->
  <section class="lm-tabpanel <?php echo ($tab==='config')?'is-visible':''; ?>" id="tab-config">
    <div class="metabox-holder lm-main" style="max-width:900px">
      <div class="postbox lm-card">
        <h2 class="hndle"><span>Ajustes</span></h2>
        <div class="inside">
          <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
            <?php wp_nonce_field('lm_save_settings','lmn_settings_nonce'); ?>
            <input type="hidden" name="action" value="lm_save_settings">
            <table class="form-table">
              <tr>
                <th><label for="lm_mail">Correo de alertas</label></th>
                <td><input id="lm_mail" name="admin_email" type="email" value="<?php echo esc_attr($settings['admin_email'] ?? get_option('admin_email')); ?>" class="regular-text"></td>
              </tr>
              <tr>
                <th><label for="lm_interval">Intervalo cron</label></th>
                <td><select id="lm_interval" name="interval">
                    <option value="fivemin"    <?php selected('fivemin',    $settings['interval'] ?? ''); ?>>Cada 5 minutos</option>
                    <option value="fifteenmin" <?php selected('fifteenmin', $settings['interval'] ?? ''); ?>>Cada 15 minutos</option>
                    <option value="hourly"     <?php selected('hourly',     $settings['interval'] ?? ''); ?>>Cada hora</option>
                    <option value="daily"      <?php selected('daily',      $settings['interval'] ?? ''); ?>>Diario</option>
                  </select></td>
              </tr>
              <tr>
                <th><label for="lm_days">Días aviso caducidad SSL</label></th>
                <td><input id="lm_days" name="days_before_expiry" type="number" min="1" value="<?php echo esc_attr($settings['days_before_expiry'] ?? 15); ?>"></td>
              </tr>
              <tr>
                <th><label for="lm_thr">Reenvío de alertas (horas)</label></th>
                <td><input id="lm_thr" name="throttle_hours" type="number" min="1" value="<?php echo esc_attr($settings['throttle_hours'] ?? 3); ?>"></td>
              </tr>
              <tr>
                <th><label for="lm_theme">Tema por defecto</label></th>
                <td><select id="lm_theme" name="default_theme">
                    <option value="system" <?php selected('system', $settings['default_theme'] ?? 'system'); ?>>Sistema</option>
                    <option value="light"  <?php selected('light',  $settings['default_theme'] ?? 'system'); ?>>Claro</option>
                    <option value="dark"   <?php selected('dark',   $settings['default_theme'] ?? 'system'); ?>>Oscuro</option>
                  </select></td>
              </tr>
              <tr>
                <th><label for="lm_allow_insec">Permitir pings sin verificación SSL</label></th>
                <td><label>
                    <input id="lm_allow_insec" type="checkbox" name="allow_insecure_pings" value="1" <?php checked( !empty($settings['allow_insecure_pings']) ); ?>>
                    <span class="description">Solo para diagnosticar. Por defecto, los pings verifican SSL.</span> </label></td>
              </tr>
            </table>
            <details class="lm-accordion">
              <summary><strong>SMTP (opcional)</strong></summary>
              <div class="lm-acc-body">
                <table class="form-table">
                  <tr>
                    <th>Activar SMTP</th>
                    <td><label>
                        <input type="checkbox" name="smtp_enabled" value="1" <?php checked(!empty($settings['smtp_enabled'])); ?>>
                        Usar SMTP para los correos de LMN Site Monitor </label></td>
                  </tr>
                  <tr>
                    <th>Servidor (host)</th>
                    <td><input type="text" name="smtp_host" value="<?php echo esc_attr($settings['smtp_host'] ?? ''); ?>" class="regular-text"></td>
                  </tr>
                  <tr>
                    <th>Puerto</th>
                    <td><input type="number" name="smtp_port" value="<?php echo esc_attr(intval($settings['smtp_port'] ?? 587)); ?>" class="small-text">
                      <span class="description">587 (TLS) o 465 (SSL), según proveedor</span></td>
                  </tr>
                  <tr>
                    <th>Cifrado</th>
                    <td><select name="smtp_secure">
                        <option value="" <?php selected(($settings['smtp_secure'] ?? '')===''); ?>>Ninguno</option>
                        <option value="tls" <?php selected(($settings['smtp_secure'] ?? '')==='tls'); ?>>TLS</option>
                        <option value="ssl" <?php selected(($settings['smtp_secure'] ?? '')==='ssl'); ?>>SSL</option>
                      </select></td>
                  </tr>
                  <tr>
                    <th>Autenticación</th>
                    <td><label>
                        <input type="checkbox" name="smtp_auth" value="1" <?php checked(!empty($settings['smtp_auth'])); ?>>
                        Requiere usuario y contraseña (recomendado) </label></td>
                  </tr>
                  <tr>
                    <th>Usuario</th>
                    <td><input type="text" name="smtp_username" value="<?php echo esc_attr($settings['smtp_username'] ?? ''); ?>" class="regular-text"></td>
                  </tr>
                  <tr>
                    <th>Contraseña</th>
                    <td><input type="password" name="smtp_password" value="" placeholder="••••••" class="regular-text">
                      <br />
                      <span class="description">Deja en blanco para mantener la contraseña guardada.</span></td>
                  </tr>
                  <tr>
                    <th>Remitente (From email)</th>
                    <td><input type="email" name="smtp_from_email" value="<?php echo esc_attr($settings['smtp_from_email'] ?? ''); ?>" class="regular-text"></td>
                  </tr>
                  <tr>
                    <th>Nombre remitente</th>
                    <td><input type="text" name="smtp_from_name" value="<?php echo esc_attr($settings['smtp_from_name'] ?? 'LMN Site Monitor'); ?>" class="regular-text"></td>
                  </tr>
                </table>
              </div>
            </details>
            <?php submit_button('Guardar ajustes','primary','lmn_settings_submit',false,['id'=>'lm-settings-save']); ?>
          </form>
        </div>
      </div>
      <div class="postbox lm-card">
        <h2 class="hndle"><span>Acerca de LMN Site Monitor</span></h2>
        <div class="inside">
          <table class="form-table">
            <tr>
              <th>Monitorización:</th>
              <td> Sitios: <strong><?php echo intval(count($sites)); ?></strong> ·
                OK: <strong><?php echo intval($ok); ?></strong> ·
                Caídos: <strong><?php echo intval($down); ?></strong> ·
                SSL caduca pronto: <strong><?php echo intval($expSoon); ?></strong></td>
            </tr>
            <tr>
              <th>Intervalo cron:</th>
              <td><?php echo esc_html($intervalLbl); ?></td>
            </tr>
            <tr>
              <th>Próxima ejecución:</th>
              <td><?php echo esc_html($nextRun); ?></td>
            </tr>
            <tr>
              <th>WP-Cron desactivado:</th>
              <td><?php echo esc_html($cronOff); ?></td>
            </tr>
            <tr>
              <th>Versión:</th>
              <td><?php echo esc_html(self::VERSION); ?></td>
            </tr>
            <tr>
              <th>Desarrollado por:</th>
              <td>Francisco Moreno Sánchez-Aguililla</td>
            </tr>
            <tr>
              <th>Web del Autor:</th>
              <td data-label="URL"><a href="https://www.fcomoreno.net" target="_blank" rel="noreferrer">www.fcomoreno.net</a></td>
            </tr>
            <tr>
              <th>Donaciones:</th>
              <td><a href="<?php echo esc_url( 'https://www.paypal.com/donate/?business=FTTYSUMRSPWKN&no_recurring=0&currency_code=EUR&item_name=Cada+donativo+marca+la+diferencia.+Tu+aportaci%C3%B3n+me+permitir%C3%A1+seguir+mejorando+LMN+Site+Monitor.+%C2%A1Muchas+Gracias%21&amount=5.00' ); ?>" title="¿Quieres hacer una donación?" target="_blank"> <img src="<?php echo esc_url( plugins_url( 'assets/donate.svg', __FILE__ ) ); ?>" alt="¿Quieres hacer una donación?" style="height: 35px"> </a></td>
            </tr>
          </table>
          <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin-top:8px;">
            <?php wp_nonce_field('lm_check_all','lm_check_all_nonce_b'); ?>
            <input type="hidden" name="action" value="lm_check_all">
            <?php submit_button('Comprobar todos ahora', 'secondary', 'lm_check_all_submit_bottom', false, array( 'id' => 'lm-check-all-bottom' )); ?>
          </form>
        </div>
      </div>
    </div>
  </section>
  <!-- Panel: Changelog -->
  <section class="lm-tabpanel <?php echo ($tab==='changelog')?'is-visible':''; ?>" id="tab-changelog">
    <div class="lm-card">
      <div class="lm-card__head">
        <h2 class="hndle" style="margin:0;">
          <?php esc_html_e('Changelog','lmn-site-monitor'); ?>
        </h2>
      </div>
      <div class="lm-card__body lm-changelog-body">
        <?php
        $changelog_path = plugin_dir_path( __FILE__ ) . 'changelog.txt';
        $raw = @file_get_contents( $changelog_path );
        if ( $raw === false || strlen( trim( $raw ) ) === 0 ) {
          echo '<p>' . esc_html__( 'No se ha encontrado el archivo changelog.txt o está vacío.', 'lmn-site-monitor' ) . '</p>';
        } else {
          // --- Markdown ligero seguro ---
          $text = str_replace( [ "\r\n", "\r" ], "\n", $raw );

          // Escape global
          $text = esc_html( $text );

          // Inline transforms: **bold**, `code`
          $text = preg_replace( '/\\*\\*(.+?)\\*\\*/s', '<strong>$1</strong>', $text );
          $text = preg_replace( '/`([^`]+)`/s', '<code>$1</code>', $text );

          // Line-based parsing
          $lines = explode( "\n", $text );
          $html = '';
          $in_list = false;

          foreach ( $lines as $line ) {
            $trim = ltrim( $line );

            // Headings starting with ###, ##, #
            if ( preg_match( '/^#{3}\s*(.+)$/', $trim, $mm ) ) {
              if ( $in_list ) {
                $html .= '</ul>';
                $in_list = false;
              }
              $html .= '<h4 style="margin:1em 0 .5em 0;">' . $mm[ 1 ] . '</h4>';
              continue;
            } elseif ( preg_match( '/^#{2}\s*(.+)$/', $trim, $mm ) ) {
              if ( $in_list ) {
                $html .= '</ul>';
                $in_list = false;
              }
              $html .= '<h3 style="margin:1.2em 0 .6em 0;">' . $mm[ 1 ] . '</h3>';
              continue;
            } elseif ( preg_match( '/^#\s*(.+)$/', $trim, $mm ) ) {
              if ( $in_list ) {
                $html .= '</ul>';
                $in_list = false;
              }
              $html .= '<h2 style="margin:1.2em 0 .6em 0;">' . $mm[ 1 ] . '</h2>';
              continue;
            }

            // Bulleted list items "- "
            if ( preg_match( '/^-\s+(.+)$/', $trim, $mm ) ) {
              if ( !$in_list ) {
                $html .= '<ul class="lm-changelog-list">';
                $in_list = true;
              }
              $item = $mm[ 1 ];
              // Highlight prefixes like NEW:, FIX:, IMPROVE:, CHANGE:, SECURITY:
              $item = preg_replace( '/^(NEW|FIX|IMPROVE|CHANGE|SECURITY):\s*/i', '<strong>$1:</strong> ', $item );
              $html .= '<li>' . $item . '</li>';
              continue;
            } else {
              if ( $in_list ) {
                $html .= '</ul>';
                $in_list = false;
              }
            }

            // Horizontal rule (=== or --- of length >=3)
            if ( preg_match( '/^(=|-){3,}\s*$/', $trim ) ) {
              $html .= '<hr class="lm-changelog-hr">';
              continue;
            }

            // Empty line -> spacer
            if ( $trim === '' ) {
              $html .= '<p style="margin:.6em 0;"></p>';
              continue;
            }

            // Default paragraph
            $html .= '<p>' . $trim . '</p>';
          }
          if ( $in_list ) {
            $html .= '</ul>';
          }

          echo wp_kses_post($html);
        }
        ?>
      </div>
    </div>
    <style>
                .lm-changelog-body {
                    padding: 16px 20px;
                    max-height: 60vh;
                    overflow: auto;
                    background: var(--lm-surface, #fff);
                    border-radius: 8px;
                }
                .lm-changelog-body pre { white-space: pre-wrap; word-wrap: break-word; font-family: Consolas, monospace; font-size: 13px; line-height: 1.5; margin: 0; }
                .lm-changelog-list { margin: 0.4em 0 0.8em 1.2em; }
                .lm-changelog-list li { margin: .2em 0; }
                .lm-changelog-hr { border: 0; height: 1px; background: var(--lm-border, #e5e7eb); margin: .8em 0; }
                .lm-changelog-body code { padding: .1em .3em; border-radius: 4px; background: rgba(0,0,0,.06); font-family: Consolas, monospace; }
            </style>
  </section>
  <style>
            /* Tabs */
            .lm-tabs { display:flex; gap:6px; margin: 12px 0 8px; border-bottom:1px solid #ccd0d4; flex-wrap:wrap;}
            .lm-tabs a { padding:6px 10px; text-decoration:none; border:1px solid transparent; border-bottom:none; border-radius:4px 4px 0 0; }
            .lm-tabs a.is-active { background:var(--wp-admin-theme-color-darker-10,#f0f0f1); border-color:#ccd0d4; }
            .lm-tabpanel { display:none; }
            .lm-tabpanel.is-visible { display:block; }

            /* Responsive grid */
            .lm-grid { display:grid; grid-template-columns: 1fr; gap:16px; }
            @media (min-width: 1024px){
                .lm-grid { grid-template-columns: 1fr 320px; }
            }
            /* Badges / days */
            .lm-badge { display:inline-block; padding:.2em .5em; border-radius:999px; font-weight:600; }
            .lm-badge.ok { background:#e7f7ed; color:#0a6d2a; }
            .lm-badge.down { background:#ff0000; color:#ffffff; }
            .lm-days.warn { font-weight:700; color: #ff3c00; }

            /* Accordion */
            details.lm-accordion { border:1px solid #ccd0d4; border-radius:6px; padding:.5rem .75rem; background:#fff; }
            details.lm-accordion > summary { cursor:pointer; outline:none; }
            details.lm-accordion .lm-acc-body { margin-top:.5rem; }
            .lm-scope.lm-dark details.lm-accordion { border: 1px solid #193a5b; background: #000000;}
        </style>
</div>
<?php
}


/* ---------- Form handlers ---------- */
public function handle_add_site() {
  if ( !current_user_can( 'manage_options' ) )wp_die( 'No autorizado' );
  check_admin_referer( 'lm_add_site', 'lm_add_site_nonce' );
  $raw_url = isset($_POST['url']) ? wp_unslash( $_POST['url'] ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- read raw input, sanitized next line
    $url = $raw_url !== '' ? esc_url_raw( trim( $raw_url ) ) : '';
  if ( !$url )wp_redirect( admin_url( 'admin.php?page=' . self::MENU_SLUG . '&msg=URL inválida' ) );
  $sites = get_option( $this->option_sites, [] );
  $id = md5( $url );
  $sites[ $id ] = array_merge( [ 'id' => $id, 'url' => $url ], $sites[ $id ] ?? [] );
  update_option( $this->option_sites, $sites );
  wp_redirect( admin_url( 'admin.php?page=' . self::MENU_SLUG . '&msg=Sitio añadido' ) );
  exit;
}

public function handle_delete_site() {
  if ( !current_user_can( 'manage_options' ) )wp_die( 'No autorizado' );
  $id = sanitize_text_field( wp_unslash( $_POST['site_id'] ?? '' ) );
  check_admin_referer( 'lm_delete_site_' . $id, 'lm_delete_nonce_' . $id );
  $sites = get_option( $this->option_sites, [] );
  if ( isset( $sites[ $id ] ) ) {
    unset( $sites[ $id ] );
    update_option( $this->option_sites, $sites );
  }
  wp_redirect( admin_url( 'admin.php?page=' . self::MENU_SLUG . '&msg=Sitio eliminado' ) );
  exit;
}

public function handle_force_check() {
  if ( !current_user_can( 'manage_options' ) )wp_die( 'No autorizado' );
  $id = sanitize_text_field( wp_unslash( $_POST['site_id'] ?? '' ) );
  check_admin_referer( 'lm_force_check_' . $id, 'lm_force_nonce_' . $id );
  $sites = get_option( $this->option_sites, [] );
  if ( isset( $sites[ $id ] ) ) {
    $this->check_site( $id, $sites[ $id ], true );
  }
  wp_redirect( admin_url( 'admin.php?page=' . self::MENU_SLUG . '&msg=Comprobado' ) );
  exit;
}

public function handle_check_all() {
  if ( !current_user_can( 'manage_options' ) )wp_die( 'No autorizado' );
  // aceptar nonce desde cualquiera de los dos formularios
  if ( isset( $_POST[ 'lm_check_all_nonce_a' ] ) ) {
    check_admin_referer( 'lm_check_all', 'lm_check_all_nonce_a' );
  } elseif ( isset( $_POST[ 'lm_check_all_nonce_b' ] ) ) {
    check_admin_referer( 'lm_check_all', 'lm_check_all_nonce_b' );
  } else {
    check_admin_referer( 'lm_check_all' ); // fallback (por si algún tema/JS reescribe el nombre)
  }

  $this->check_all_sites_cron();
  wp_redirect( admin_url( 'admin.php?page=' . self::MENU_SLUG . '&msg=Comprobación lanzada' ) );
  exit;
}

public function handle_save_settings() {
  if ( ! current_user_can( 'manage_options' ) ) {
    wp_die( esc_html__( 'No autorizado', 'lmn-site-monitor' ) );
  }
  check_admin_referer( 'lm_save_settings', 'lmn_settings_nonce' );

  $allowed_intervals = [ 'fivemin', 'fifteenmin', 'hourly', 'daily' ];
  $old = get_option( $this->option_settings, [] );

  // Intervalo
  $post_interval = isset( $_POST['interval'] ) ? sanitize_key( wp_unslash( $_POST['interval'] ) ) : ( $old['interval'] ?? 'hourly' );
  $interval = in_array( $post_interval, $allowed_intervals, true ) ? $post_interval : ( $old['interval'] ?? 'hourly' );

  // Construir nuevo set de opciones
  $new = [
    'admin_email'        => isset( $_POST['admin_email'] ) ? sanitize_email( wp_unslash( $_POST['admin_email'] ) ) : ( $old['admin_email'] ?? get_option( 'admin_email' ) ),
    'interval'           => $interval,
    'days_before_expiry' => max( 1, absint( isset( $_POST['days_before_expiry'] ) ? wp_unslash( $_POST['days_before_expiry'] ) : ( $old['days_before_expiry'] ?? 15 ) ) ),
    'throttle_hours'     => max( 1, absint( isset( $_POST['throttle_hours'] ) ? wp_unslash( $_POST['throttle_hours'] ) : ( $old['throttle_hours'] ?? 3 ) ) ),
    'default_theme'      => ( function() use ( $old ) {
      $val = isset( $_POST['default_theme'] ) ? sanitize_key( wp_unslash( $_POST['default_theme'] ) ) : ( $old['default_theme'] ?? 'system' ); // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above
      return in_array( $val, [ 'system', 'light', 'dark' ], true ) ? $val : 'system';
    } )(),
    'use_local_font'       => ! empty( $_POST['use_local_font'] ) ? 1 : 0,
    'allow_insecure_pings' => ! empty( $_POST['allow_insecure_pings'] ) ? 1 : 0,
  ];

  // SMTP
  $new['smtp_enabled']   = ! empty( $_POST['smtp_enabled'] ) ? 1 : 0;
  $new['smtp_host']      = isset( $_POST['smtp_host'] ) ? sanitize_text_field( wp_unslash( $_POST['smtp_host'] ) ) : ( $old['smtp_host'] ?? '' );
  $new['smtp_port']      = isset( $_POST['smtp_port'] ) ? absint( wp_unslash( $_POST['smtp_port'] ) ) : ( $old['smtp_port'] ?? 587 );
  $new['smtp_secure']    = ( function () use ( $old ) {
    $raw = isset( $_POST['smtp_secure'] ) ? sanitize_key( wp_unslash( $_POST['smtp_secure'] ) ) : ( $old['smtp_secure'] ?? '' ); // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above
    $allowed = [ '', 'ssl', 'tls' ];
    return in_array( $raw, $allowed, true ) ? $raw : '';
  } )();
  $new['smtp_auth']      = ! empty( $_POST['smtp_auth'] ) ? 1 : 0;
  $new['smtp_username']  = isset( $_POST['smtp_username'] ) ? sanitize_text_field( wp_unslash( $_POST['smtp_username'] ) ) : ( $old['smtp_username'] ?? '' );
  // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Password may contain any chars
  $new['smtp_password']  = isset( $_POST['smtp_password'] ) ? (string) wp_unslash( $_POST['smtp_password'] ) : ( $old['smtp_password'] ?? '' );
  $new['smtp_from_email']= isset( $_POST['smtp_from_email'] ) ? sanitize_email( wp_unslash( $_POST['smtp_from_email'] ) ) : ( $old['smtp_from_email'] ?? '' );
  $new['smtp_from_name'] = isset( $_POST['smtp_from_name'] ) ? sanitize_text_field( wp_unslash( $_POST['smtp_from_name'] ) ) : ( $old['smtp_from_name'] ?? 'LMN Site Monitor' );

  update_option( $this->option_settings, $new );

  // Reprogramar cron si cambió el intervalo
  if ( ( $old['interval'] ?? 'hourly' ) !== $new['interval'] ) {
    wp_clear_scheduled_hook( self::CRON_HOOK );
    $schedules = function_exists( 'wp_get_schedules' ) ? wp_get_schedules() : [];
    $sched = isset( $schedules[ $new['interval'] ] ) ? $new['interval'] : 'hourly';
    wp_schedule_event( time() + 60, $sched, self::CRON_HOOK );
  }

  wp_redirect( add_query_arg( [ 'page' => self::MENU_SLUG, 'msg' => 'Ajustes guardados' ], admin_url( 'admin.php' ) ) );
  exit;
}


// Preferencia del widget: cuántos sitios mostrar
public function handle_save_widget_prefs() {
  if ( !current_user_can( 'read' ) )wp_die( 'No autorizado' );
  check_admin_referer( 'lm_save_widget_prefs', 'lm_widget_nonce' );
  $limit = isset( $_POST[ 'lm_widget_limit' ] ) ? intval( $_POST[ 'lm_widget_limit' ] ) : 5;
  if ( $limit < 1 )$limit = 1;
  if ( $limit > 100 )$limit = 100;
  update_user_meta( get_current_user_id(), 'lm_widget_limit', $limit );
  wp_redirect( admin_url( 'index.php?lm_widget_saved=1' ) );
  exit;
}

// NUEVO: Exportación CSV
public function handle_export_csv() {
  if ( !current_user_can( 'manage_options' ) )wp_die( 'No autorizado' );
  check_admin_referer( 'lm_export_csv', 'lm_export_nonce' );

  $sites = get_option( $this->option_sites, [] );
  if ( !is_array( $sites ) )$sites = [];

  // Orden: caídos primero, luego por URL
  uasort( $sites, function ( $a, $b ) {
    $sa = $a[ 'last_status' ] ?? '';
    $sb = $b[ 'last_status' ] ?? '';
    if ( $sa === 'down' && $sb !== 'down' ) return -1;
    if ( $sb === 'down' && $sa !== 'down' ) return 1;
    return strcmp( $a[ 'url' ] ?? '', $b[ 'url' ] ?? '' );
  } );

  $filename = 'site-monitor-export-' . date_i18n( 'Ymd-His' ) . '.csv';
  nocache_headers();
  header( 'Content-Type: text/csv; charset=UTF-8' );
  header( 'Content-Disposition: attachment; filename="' . $filename . '"' );

  $out = fopen( 'php://output', 'w' );
  // BOM UTF-8 para compatibilidad con Excel
  fprintf( $out, "\xEF\xBB\xBF" );

  // Encabezados
  fputcsv( $out, [ 'URL', 'Última comprobación', 'Estado', 'Código HTTP', 'SSL caduca', 'Días restantes SSL' ] );

  foreach ( $sites as $s ) {
    $url = $s[ 'url' ] ?? '';
    $checked = !empty( $s[ 'last_checked' ] ) ? date_i18n( 'Y-m-d H:i:s', $s[ 'last_checked' ] ) : '';
    $status = $s[ 'last_status' ] ?? '';
    $statusL = ( $status === 'down' ? 'Caído' : ( $status === 'ok' ? 'OK' : '' ) );
    $http = isset( $s[ 'http_code' ] ) ? ( string )$s[ 'http_code' ] : '';
    $sslF = !empty( $s[ 'ssl_expires' ] ) ? date_i18n( 'Y-m-d H:i', $s[ 'ssl_expires' ] ) : '';
    $sslD = isset( $s[ 'ssl_expires' ] ) ? ceil( ( $s[ 'ssl_expires' ] - time() ) / 86400 ) : '';

    fputcsv( $out, [ $url, $checked, $statusL, $http, $sslF, $sslD ] );
  }

  // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- php://output does not need closing
// fclose( $out );
  exit;
}

/* ---------- Dashboard widget ---------- */
public function dashboard_widget() {
  wp_add_dashboard_widget( 'lm_monitor_widget', 'Resumen Site Monitor', [ $this, 'render_dashboard_widget' ] );
}



  public function handle_import_csv() {
    if ( ! current_user_can( 'manage_options' ) ) { wp_die( esc_html__( 'No autorizado', 'lmn-site-monitor' ) ); }
    check_admin_referer( 'lm_import_csv', 'lm_import_nonce' );
        // Carga robusta del CSV desde el archivo temporal subido
        if ( empty( $_FILES['csv_file'] ) || empty( $_FILES['csv_file']['tmp_name'] ) ) {
            wp_redirect( admin_url( 'admin.php?page=' . self::MENU_SLUG . '&msg=No+se+ha+subido+ning%C3%BAn+CSV' ) );
            exit;
        }
        $tmp = isset($_FILES['csv_file']['tmp_name']) ? $_FILES['csv_file']['tmp_name'] : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- tmp_name from PHP, validated via is_uploaded_file()
          // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- tmp_name is a PHP-provided path; verified with is_uploaded_file
          if ( ! $tmp || ! is_uploaded_file( $tmp ) ) {
              wp_redirect( admin_url( 'admin.php?page=' . self::MENU_SLUG . '&msg=No+se+pudo+leer+el+CSV' ) );
              exit;
          }
        $content = @file_get_contents( $tmp );
        if ( $content === false ) {
            global $wp_filesystem;
            if ( ! $wp_filesystem ) {
                require_once ABSPATH . 'wp-admin/includes/file.php';
                WP_Filesystem();
            }
            if ( $wp_filesystem ) {
                $content = $wp_filesystem->get_contents( $tmp );
            }
        }
        if ( $content === false || $content === '' ) {
            wp_redirect( admin_url( 'admin.php?page=' . self::MENU_SLUG . '&msg=No+se+pudo+leer+el+CSV' ) );
            exit;
        }
        // Quitar BOM UTF-8 si existe
        if ( substr( $content, 0, 3 ) === "\xEF\xBB\xBF" ) {
            $content = substr( $content, 3 );
        }
        // Normalizar saltos de línea
        $content = str_replace( array("\r\n", "\r"), "\n", $content );
        $lines   = explode( "\n", $content );

    if ( ! current_user_can( 'manage_options' ) ) wp_die( 'No autorizado' );
    check_admin_referer( 'lm_import_csv', 'lm_import_nonce' );

    if ( empty( $_FILES['csv_file'] ) || ! isset( $_FILES['csv_file']['tmp_name'] ) ) {
      wp_redirect( admin_url( 'admin.php?page=' . self::MENU_SLUG . '&msg=Sin archivo' ) );
      exit;
    }

    $file = $_FILES['csv_file']; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- WP handles upload superglobal after nonce
    if ( ! empty( $file['error'] ) && $file['error'] !== UPLOAD_ERR_OK ) {
      wp_redirect( admin_url( 'admin.php?page=' . self::MENU_SLUG . '&msg=Error subiendo CSV (' . intval( $file['error'] ) . ')' ) );
      exit;
    }

    $tmp = $file['tmp_name'];
    if ( ! is_uploaded_file( $tmp ) && ! file_exists( $tmp ) ) {
      wp_redirect( admin_url( 'admin.php?page=' . self::MENU_SLUG . '&msg=No se pudo leer el CSV' ) );
      exit;
    }

    $content = @file_get_contents( $tmp );
    if ( $content === false ) {
      global $wp_filesystem;
      if ( ! $wp_filesystem ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        WP_Filesystem();
      }
      if ( $wp_filesystem ) {
        $content = $wp_filesystem->get_contents( $tmp );
      }
    }

    if ( $content === false || $content === '' ) {
      wp_redirect( admin_url( 'admin.php?page=' . self::MENU_SLUG . '&msg=No se pudo leer el CSV' ) );
      exit;
    }

    if ( substr( $content, 0, 3 ) === "\xEF\xBB\xBF" ) {
      $content = substr( $content, 3 );
    }

    $lines = preg_split( '/\r\n|\r|\n/', $content );

    $sample = '';
    foreach ( $lines as $__l ) { if ( trim($__l) !== '' ) { $sample = $__l; break; } }
    $delims = [ ',', ';', "\t" ];
    $best_delim = ',';
    $best_count = -1;
    foreach ( $delims as $d ) {
      $c = substr_count( $sample, $d );
      if ( $c > $best_count ) { $best_count = $c; $best_delim = $d; }
    }

    $sites = get_option( $this->option_sites, [] );
    $added = 0;
    $seen  = [];

    foreach ( $lines as $__csv_line ) {
      $__csv_line = trim( $__csv_line );
      if ( $__csv_line === '' ) continue;

      $row = str_getcsv( $__csv_line, $best_delim );
      if ( empty( $row ) || $row === [ null ] ) continue;

      $url = isset( $row[0] ) ? trim( (string) $row[0] ) : '';
      if ( $url === '' ) continue;
      if ( strtolower( $url ) === 'url' ) continue;

      $url = esc_url_raw( $url );
      if ( ! $url ) continue;

      if ( isset( $seen[ $url ] ) ) continue;
      $seen[ $url ] = true;

      $id = md5( $url );
      if ( ! isset( $sites[ $id ] ) ) {
        $sites[ $id ] = [ 'id' => $id, 'url' => $url ];
        $added++;
      }
    }

    if ( $added > 0 ) {
      update_option( $this->option_sites, $sites );
    }

    wp_redirect( admin_url( 'admin.php?page=' . self::MENU_SLUG . '&msg=Importados+' . intval( $added ) ) );
    exit;
  }


public function handle_bulk_action() {
  if ( !current_user_can( 'manage_options' ) )wp_die( 'No autorizado' );
  check_admin_referer( 'lm_bulk_action', 'lm_bulk_nonce' );
  $action = sanitize_text_field( wp_unslash( $_POST['bulk_action'] ?? '' ) );
  $ids = ( isset($_POST['ids']) && is_array($_POST['ids']) ) ? array_map( 'sanitize_text_field', wp_unslash( $_POST['ids'] ) ) : [];
  if ( $action === 'delete' && $ids ) {
    $sites = get_option( $this->option_sites, [] );
    $removed = 0;
    foreach ( $ids as $id ) {
      if ( isset( $sites[ $id ] ) ) {
        unset( $sites[ $id ] );
        $removed++;
      }
    }
    if ( $removed > 0 )update_option( $this->option_sites, $sites );
    wp_redirect( admin_url( 'admin.php?page=' . self::MENU_SLUG . '&msg=Eliminados+' . intval( $removed ) ) );
    exit;
  }
  wp_redirect( admin_url( 'admin.php?page=' . self::MENU_SLUG ) );
  exit;
}
public function render_dashboard_widget() {
  $sites = get_option( $this->option_sites, [] );
  $settings = get_option( $this->option_settings, [] );

  // Preferencia por usuario: límite de filas
  $user_id = get_current_user_id();
  $limit = intval( get_user_meta( $user_id, 'lm_widget_limit', true ) );
  if ( $limit <= 0 )$limit = 5;

  $ok = 0;
  $down = 0;
  $expSoon = 0;
  $rows = [];
  $days_before = intval( $settings[ 'days_before_expiry' ] );
  foreach ( $sites as $s ) {
    $status = $s[ 'last_status' ] ?? '';
    if ( $status === 'down' )$down++;
    else $ok++;
    if ( !empty( $s[ 'ssl_expires' ] ) ) {
      $d = ceil( ( $s[ 'ssl_expires' ] - time() ) / 86400 );
      if ( $d <= $days_before )$expSoon++;
    }
    $rows[] = [
      'url' => $s[ 'url' ],
      'status' => $status,
      'checked' => isset( $s[ 'last_checked' ] ) ? date_i18n( 'Y-m-d H:i', $s[ 'last_checked' ] ) : '—',
    ];
  }
  usort( $rows, function ( $a, $b ) {
    if ( $a[ 'status' ] === 'down' && $b[ 'status' ] !== 'down' ) return -1;
    if ( $b[ 'status' ] === 'down' && $a[ 'status' ] !== 'down' ) return 1;
    return strcmp( $a[ 'url' ], $b[ 'url' ] );
  } );
  $rows = array_slice( $rows, 0, $limit );

  echo '<div class="lm-scope">';
  echo '<style>
            .lm-kpis{display:flex;gap:8px;margin-bottom:8px}
            .lm-kpis .k{flex:1 1 0;background:var(--lm-surface);border:1px solid var(--lm-border);border-radius:10px;padding:8px;text-align:center;color:var(--lm-text);box-shadow:var(--lm-shadow)}
            .lm-mini{width:100%;border-collapse:collapse}
            .lm-mini td{padding:6px 8px;border-bottom:1px solid var(--lm-border);color:var(--lm-text)}
            .lm-mini .dot{width:10px;height:10px;border-radius:50%;display:inline-block;margin-right:6px}
            .lm-mini .dot.ok{background:#16a34a}.lm-mini .dot.down{background:#dc2626}
            .lm-mini .right{text-align:right;color:var(--lm-muted)}
            .lm-widget-pref{display:flex;gap:6px;align-items:center;justify-content:flex-end;margin:6px 0;}
            .lm-widget-pref label{font-size:12px;color:var(--lm-muted)}
            .lm-widget-pref select{min-width:74px}
            .lm-widget-pref .button{padding:2px 8px;line-height:1.6;height:auto}
        </style>';

  // Preferencia de límite (form en el propio widget)
  echo '<form class="lm-widget-pref" method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '">';
  wp_nonce_field( 'lm_save_widget_prefs', 'lm_widget_nonce' );
  echo '<input type="hidden" name="action" value="lm_save_widget_prefs">';
  echo '<label for="lm_widget_limit">Ver</label>';
  echo '<select id="lm_widget_limit" name="lm_widget_limit">';
  foreach ( [ 5, 10, 15, 20, 30, 50 ] as $opt ) {
    echo '<option value="' . esc_attr( $opt ) . '" ' . selected( $limit, $opt, false ) . '>' . esc_html( $opt ) . '</option>';
  }
  echo '</select> <span style="font-size:12px;color:var(--lm-muted)">sitios</span>';
  submit_button( 'Aplicar', 'secondary', 'lm_widget_apply', false, [ 'id' => 'lm-widget-apply' ] );
  echo '</form>';

  // KPIs
  echo '<div class="lm-kpis">
                <div class="k"><strong>OK</strong><div>' . intval( $ok ) . '</div></div>
                <div class="k"><strong>Caídos</strong><div>' . intval( $down ) . '</div></div>
                <div class="k"><strong>SSL a caducar</strong><div>' . intval( $expSoon ) . '</div></div>
              </div>';

  // Listado compacto
  if ( !empty( $rows ) ) {
    // La impresión de <table> es segura porque es HTML estático.
    echo '<table class="lm-mini"><tbody>';
    foreach ( $rows as $r ) {
      $dot = $r[ 'status' ] === 'down' ? 'down' : 'ok';
      echo '<tr>
                <td class="dot ' . esc_attr( $dot ) . '"></td> <td data-label="URL"><a href="' . esc_url( $r[ 'url' ] ) . '" target="_blank" rel="noreferrer">' . esc_html( $r[ 'url' ] ) . '</a></td>
                <td class="right">' . esc_html( $r[ 'checked' ] ) . '</td> </tr>';
    }
    echo '</tbody></table>';
  } else {
    // HTML estático seguro. Se podría usar esc_html_e() si fuera texto traducible.
    echo '<p>No hay sitios configurados.</p>';
  }
  echo '<p style="margin-top:8px"><a class="button button-primary" href="' . esc_url( admin_url( 'admin.php?page=' . self::MENU_SLUG ) ) . '">Abrir Site Monitor</a></p>'; // esc_url ya estaba correcto aquí
  echo '</div>';
}

/* ---------- Core checks ---------- */
public function check_all_sites_cron() {
  $sites = get_option( $this->option_sites, [] );
  foreach ( $sites as $id => $s )$this->check_site( $id, $s );
}

public function check_site( $id, $site, $force_email = false ) {
  $url = rtrim( $site[ 'url' ], '/' );
  $result = [
    'id' => $id,
    'url' => $url,
    'last_checked' => time(),
    'last_status' => 'ok',
    'http_code' => null,
    'ssl_expires' => null,
    'ssl_days_left' => null,
  ];

  // Ping HTTP(S)
  $response = wp_remote_get( $url, [ 'timeout' => 15, 'redirection' => 5, 'sslverify' => !( bool )get_option( 'lmn_ssl_insecure_pings', 0 ), 'method' => 'HEAD' ] );
  if ( is_wp_error( $response ) ) {
    $result[ 'last_status' ] = 'down';
    $result[ 'http_code' ] = 0;
    $result[ 'error_msg' ] = $response->get_error_message();
    $result[ 'http_code' ] = 0;
  } else {
    $code = wp_remote_retrieve_response_code( $response );
    $msg = wp_remote_retrieve_response_message( $response );
    $result[ 'http_code' ] = $code;
    $result[ 'status_label' ] = trim( $code . ' ' . ( $msg ? : '' ) );
    if ( $code >= 400 ) {
      $result[ 'last_status' ] = 'down';
      $body = wp_remote_retrieve_body( $response );
      if ( $body ) {
        $snippet = wp_strip_all_tags( mb_substr( $body, 0, 500 ) );
        $result[ 'error_msg' ] = ( $result[ 'status_label' ] ?? '' ) . ' — ' . $snippet;
      }
    }
  }

  // SSL (solo https)
  $parsed = wp_parse_url( $url );
  if ( ( $parsed[ 'scheme' ] ?? '' ) === 'https' && !empty( $parsed[ 'host' ] ) ) {
    $cert_ts = $this->get_ssl_expiry_timestamp( $parsed[ 'host' ] );
    if ( $cert_ts !== false ) {
      $result[ 'ssl_expires' ] = $cert_ts;
      $result[ 'ssl_days_left' ] = ceil( ( $cert_ts - time() ) / 86400 );
    }
  }

  // Downtime tracking
  $prev_status = isset( $site[ 'last_status' ] ) ? $site[ 'last_status' ] : 'unknown';
  if ( ( $result[ 'last_status' ] ?? '' ) === 'down' && $prev_status !== 'down' ) {
    $result[ 'downtime_start' ] = time();
  } elseif ( ( $result[ 'last_status' ] ?? '' ) === 'ok' && $prev_status === 'down' ) {
    // keep previous downtime_start in $site if exists
    if ( !empty( $site[ 'downtime_start' ] ) )$result[ 'downtime_start' ] = $site[ 'downtime_start' ];
  }

  // Guardar
  $sites = get_option( $this->option_sites, [] );
  $sites[ $id ] = array_merge( $site, $result );
  update_option( $this->option_sites, $sites );

  // Alertas
  $this->maybe_send_alert( $id, $sites[ $id ], $force_email );
}

private function get_ssl_expiry_timestamp( $host ) {
  if ( empty( $host ) ) return false;
  $context = stream_context_create( [
    'ssl' => [
      'capture_peer_cert' => true,
      'verify_peer' => false,
      'verify_peer_name' => false,
      'SNI_enabled' => true,
      'peer_name' => $host,
      'crypto_method' => ( defined( 'STREAM_CRYPTO_METHOD_TLSv1_3_CLIENT' ) ? STREAM_CRYPTO_METHOD_TLSv1_2_CLIENT | STREAM_CRYPTO_METHOD_TLSv1_3_CLIENT : STREAM_CRYPTO_METHOD_TLSv1_2_CLIENT ),
    ]
  ] );
  $client = @stream_socket_client( "ssl://{$host}:443", $errno, $errstr, 15, STREAM_CLIENT_CONNECT, $context );
  if ( !$client ) return false;
  $params = stream_context_get_params( $client );
  if ( empty( $params[ 'options' ][ 'ssl' ][ 'peer_certificate' ] ) ) return false;
  $certinfo = openssl_x509_parse( $params[ 'options' ][ 'ssl' ][ 'peer_certificate' ] );
  if ( !$certinfo || empty( $certinfo[ 'validTo_time_t' ] ) ) return false;
  return intval( $certinfo[ 'validTo_time_t' ] );
}

/* ---------- Email bonito (HTML claro por defecto + dark automático) ---------- */
private function build_alert_email( array $siteData, array $alerts ): array {
  $host = wp_parse_url( $siteData[ 'url' ], PHP_URL_HOST ) ? : $siteData[ 'url' ];
  $hasDown = array_reduce( $alerts, fn( $c, $a ) => $c || $a[ 'type' ] === 'down', false );
  $hasSSL = array_reduce( $alerts, fn( $c, $a ) => $c || $a[ 'type' ] === 'ssl_expiring', false );

  // Asunto
  if ( $hasDown )$subject = "⛔ Sitio caído: {$host}";
  elseif ( $hasSSL )$subject = "⚠️ SSL por caducar: {$host}";
  else $subject = "🔔 Aviso sitio: {$host}";

  // Datos
  $checked = gmdate( 'Y-m-d H:i:s', $siteData[ 'last_checked' ] );
  $http = $siteData[ 'http_code' ] ?? '—';
  $err = isset( $siteData[ 'error_msg' ] ) ? $siteData[ 'error_msg' ] : '';
  $sslDate = !empty( $siteData[ 'ssl_expires' ] ) ? gmdate( 'Y-m-d H:i', $siteData[ 'ssl_expires' ] ) : '—';
  $sslDays = $siteData[ 'ssl_days_left' ] ?? null;

  // URLs útiles
  $adminUrl = esc_url( admin_url( 'admin.php?page=' . self::MENU_SLUG ) );
  $siteUrl = esc_url( $siteData[ 'url' ] );

  // Ajustes para el pie
  $settings = get_option( $this->option_settings, [] );
  $admin_to = $settings[ 'admin_email' ] ?? get_option( 'admin_email' );
  $days_before = intval( $settings[ 'days_before_expiry' ] ?? 5 );
  $throttle_hours = intval( $settings[ 'throttle_hours' ] ?? 12 );

  // Listado de incidencias
  $li = '';
  foreach ( $alerts as $a ) {
    $li .= '<li class="m-li" style="margin:6px 0;">' . esc_html( $a[ 'message' ] ) . '</li>';
  }

  // Píldora días SSL
  $daysPill = ( $sslDays !== null ) ?
    '<span class="m-pill" style="display:inline-block;padding:3px 8px;border-radius:6px;background:#eef2f7;color:#111;font:600 12px/1 system-ui;">' . intval( $sslDays ) . ' días</span>': '—';

  // Badge de estado
  $statusBadge = ( $siteData[ 'last_status' ] ?? '' ) === 'down' ?
    '<span style="display:inline-block;padding:3px 10px;border-radius:999px;background:#dc2626;color:#fff;font:700 12px/1 system-ui;">Caído</span>' :
    '<span style="display:inline-block;padding:3px 10px;border-radius:999px;background:#16a34a;color:#fff;font:700 12px/1 system-ui;">OK</span>';

  // HTML completo (light base + dark overrides por media query)
  $html = '<!doctype html>
<html>
<head>
<meta charset="utf-8">
<meta name="color-scheme" content="light dark">
<meta name="supported-color-schemes" content="light dark">
<style>
  /* Dark overrides: solo en clientes que soportan prefers-color-scheme */
  @media (prefers-color-scheme: dark) {
    .m-wrap { background:#0f141b !important; color:#e5e7eb !important; }
    .m-card { background:#101720 !important; border-color:#1f2937 !important; }
    .m-head { color:#ffffff !important; }
    .m-txt  { color:#cbd5e1 !important; }
    .m-table-hd { background:#0e141d !important; }
    .m-th { color:#9fb3c8 !important; border-bottom:1px solid #233143 !important; }
    .m-td { color:#e5e7eb !important; border-bottom:1px solid #233143 !important; }
    .m-li { color:#e5e7eb !important; }
    .m-btn-pri { background:#3b82f6 !important; color:#fff !important; }
    .m-btn-sec { background:#1f2937 !important; color:#e5e7eb !important; }
    .m-pill { background:#2b3443 !important; color:#e5e7eb !important; }
    a { color:#93c5fd !important; }
  }
</style>
</head>
<body class="m-wrap" style="margin:0;padding:0;background:#f5f7fb;color:#0f172a;">
  <div style="height:24px;"></div>
  <div class="m-card" style="max-width:640px;margin:0 auto;background:#ffffff;border:1px solid #e5e7eb;border-radius:12px;overflow:hidden;">
    <div style="padding:16px 20px;">
      <h1 class="m-head" style="font:600 18px/1.3 system-ui,Segoe UI,Roboto,Helvetica,Arial;color:#0f172a;margin:0;">Monitor de sitios · ' . $this->esc_for_email( $host ) . '</h1>
      <p class="m-txt" style="font:400 14px/1.6 system-ui,Segoe UI,Roboto,Helvetica,Arial;margin:6px 0 0;color:#334155;">Se han detectado incidencias en la última comprobación.</p>
    </div>

    <div class="m-table-hd" style="background:#f8fafc;padding:16px 20px;">
      <ul style="margin:0;padding-left:18px;color:inherit;font:400 14px/1.6 system-ui;">' . $li . '</ul>

      <table style="width:100%;border-collapse:collapse;margin-top:10px;">
        <thead>
          <tr>
            <th class="m-th" style="text-align:left;padding:10px;border-bottom:1px solid #e5e7eb;color:#334155;font:600 12px/1 system-ui;">Campo</th>
            <th class="m-th" style="text-align:left;padding:10px;border-bottom:1px solid #e5e7eb;color:#334155;font:600 12px/1 system-ui;">Valor</th>
          </tr>
        </thead>
        <tbody>
          <tr><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#0f172a;font:400 13px/1.4 system-ui;">Estado</td><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#0f172a;font:400 13px/1.4 system-ui;">' . $statusBadge . '</td></tr>
          <tr><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#0f172a;font:400 13px/1.4 system-ui;">Código HTTP</td><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#0f172a;font:400 13px/1.4 system-ui;">' . esc_html( $http ) . '</td></tr>
          <tr><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#0f172a;font:400 13px/1.4 system-ui;">Última comprobación</td><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#0f172a;font:400 13px/1.4 system-ui;">' . esc_html( $checked ) . '</td></tr>
          <tr><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#0f172a;font:400 13px/1.4 system-ui;">SSL caduca</td><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#0f172a;font:400 13px/1.4 system-ui;">' . esc_html( $sslDate ) . '</td></tr>
          <tr><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#0f172a;font:400 13px/1.4 system-ui;">Días restantes SSL</td><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#0f172a;font:400 13px/1.4 system-ui;">' . $daysPill . '</td></tr>
          <tr><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#0f172a;font:400 13px/1.4 system-ui;">URL</td><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#0f172a;font:400 13px/1.4 system-ui;"><a href="' . $siteUrl . '" style="color:#2563eb;text-decoration:none;">' . $siteUrl . '</a></td></tr>
        </tbody>
      </table>

      <div style="margin-top:16px;display:flex;gap:10px;">
        <a class="m-btn-pri" href="' . $siteUrl . '" style="display:inline-block;background:#2563eb;color:#fff;text-decoration:none;border-radius:8px;padding:10px 14px;font:600 14px/1 system-ui;" target="_blank" rel="noreferrer">Abrir sitio</a>
        <a class="m-btn-sec" href="' . $adminUrl . '" style="display:inline-block;background:#e5e7eb;color:#111827;text-decoration:none;border-radius:8px;padding:10px 14px;font:600 14px/1 system-ui;">Ver en Site Monitor</a>
      </div>
    </div>

    <div style="padding:16px 20px;background:#ffffff;color:#64748b;font:400 12px/1.6 system-ui;">
      Este correo se envía a ' . $this->esc_for_email( $admin_to ) . '.<br>
      Umbral SSL: ' . $days_before . ' días · Reenvío cada ' . $throttle_hours . ' h.
    </div>
  </div>
  <div style="height:24px;"></div>
</body>
</html>';

  return [ $subject, $html ];
}

private function esc_for_email( $str ) {
  return esc_html( $str ?? '' );
}

private function maybe_send_alert( $id, $siteData, $force_email = false ) {
  $settings = get_option( $this->option_settings, [] );
  $admin_email = $settings[ 'admin_email' ] ?? get_option( 'admin_email' );
  $days_before = intval( $settings[ 'days_before_expiry' ] ?? 5 );
  $throttle_seconds = max( 3600, intval( $settings[ 'throttle_hours' ] ?? 12 ) * 3600 );

  $last_alerts = get_option( $this->option_last_alerts, [] );

  $alerts = [];
  if ( ( $siteData[ 'last_status' ] ?? '' ) === 'down' ) {
    $alerts[] = [ 'type' => 'down', 'message' => "Sitio caído: {$siteData['url']} (HTTP: {$siteData['http_code']})." ];
  }
  if ( !empty( $siteData[ 'ssl_expires' ] ) ) {
    $days = ceil( ( $siteData[ 'ssl_expires' ] - time() ) / 86400 );
    if ( $days <= $days_before ) {
      $alerts[] = [ 'type' => 'ssl_expiring', 'message' => "Certificado caduca en {$days} día(s) para {$siteData['url']} (fecha: " . gmdate( 'Y-m-d H:i:s', $siteData[ 'ssl_expires' ] ) . ")." ];
    }
  }

  if ( empty( $alerts ) ) {
    if ( isset( $last_alerts[ $id ] ) && $last_alerts[ $id ][ 'state' ] !== 'ok' ) {
      // Send recovered email once
      [ $subject, $html ] = $this->build_recovered_email( $siteData );
      $headers = [ 'Content-Type: text/html; charset=UTF-8' ];
      wp_mail( $admin_email, "[Monitor] " . $subject, $html, $headers );
      $last_alerts[ $id ] = [ 'state' => 'ok', 'sent' => 0, 'last_sent' => time() ];
      update_option( $this->option_last_alerts, $last_alerts );
    }
    return;
  }

  // Throttle/envío
  $send = false;
  if ( $force_email )$send = true;
  if ( !isset( $last_alerts[ $id ] ) ) {
    $send = true;
    $last_alerts[ $id ] = [ 'state' => 'problem', 'sent' => 0, 'last_sent' => 0 ];
  } else {
    if ( $last_alerts[ $id ][ 'state' ] === 'ok' )$send = true;
    else if ( time() - intval( $last_alerts[ $id ][ 'last_sent' ] ) > $throttle_seconds )$send = true;
  }

  if ( $send ) {
    // Email HTML claro + dark automático
    [ $subject, $html ] = $this->build_alert_email( $siteData, $alerts );
    $headers = [ 'Content-Type: text/html; charset=UTF-8' ];
    wp_mail( $admin_email, "[Monitor] " . $subject, $html, $headers );

    $last_alerts[ $id ] = [
      'state' => 'problem',
      'sent' => intval( $last_alerts[ $id ][ 'sent' ] ?? 0 ) + 1,
      'last_sent' => time()
    ];
    update_option( $this->option_last_alerts, $last_alerts );
  }
}

public function build_recovered_email( array $siteData ): array {
  $host = wp_parse_url( $siteData[ 'url' ], PHP_URL_HOST ) ? : $siteData[ 'url' ];
  $subject = "🟢 Sitio recuperado: {$host}";

  // Datos
  $checked = gmdate( 'Y-m-d H:i:s', $siteData[ 'last_checked' ] ?? time() );
  $http = $siteData[ 'http_code' ] ?? '—';
  $sslDate = !empty( $siteData[ 'ssl_expires' ] ) ? gmdate( 'Y-m-d H:i', $siteData[ 'ssl_expires' ] ) : '—';
  $sslDays = $siteData[ 'ssl_days_left' ] ?? null;

  // URLs útiles
  $adminUrl = esc_url( admin_url( 'admin.php?page=' . self::MENU_SLUG ) );
  $siteUrl = esc_url( $siteData[ 'url' ] );

  // Ajustes para el pie
  $settings = get_option( $this->option_settings, [] );
  $admin_to = $settings[ 'admin_email' ] ?? get_option( 'admin_email' );
  $days_before = intval( $settings[ 'days_before_expiry' ] ?? 5 );
  $throttle_hours = intval( $settings[ 'throttle_hours' ] ?? 12 );

  // Periodo de caída (si existe)
  $downtimeFrom = !empty( $siteData[ 'downtime_start' ] ) ? gmdate( 'Y-m-d H:i', intval( $siteData[ 'downtime_start' ] ) ) : '—';

  // Píldora días SSL
  $daysPill = ( $sslDays !== null ) ?
    '<span class="m-pill" style="display:inline-block;padding:3px 8px;border-radius:999px;background:#e2e8f0;color:#0f172a;font:600 12px/1 system-ui;">' . intval( $sslDays ) . ' días</span>': '—';

  // Badge de estado (OK)
  $statusBadge = '<span style="display:inline-block;padding:3px 10px;border-radius:6px;background:#16a34a;color:#fff;font:700 12px/1 system-ui;">OK</span>';

  // Lista de mensajes (idéntica maquetación que alertas)
  $li = '<li class="m-li" style="margin:6px 0;">El sitio vuelve a estar operativo.</li>';
  if ( $downtimeFrom !== '—' ) {
    $li .= '<li class="m-li" style="margin:6px 0;">Estuvo caído desde: ' . esc_html( $downtimeFrom ) . '</li>';
  }

  // HTML unificado con el de alertas (light + dark automático)
  $html = '<!doctype html>
<html>
<head>
<meta charset="utf-8">
<meta name="color-scheme" content="light dark">
<meta name="supported-color-schemes" content="light dark">
<style>
  /* Dark overrides: solo en clientes que soportan prefers-color-scheme */
  @media (prefers-color-scheme: dark) {
    .m-wrap { background:#0f141b !important; color:#e5e7eb !important; }
    .m-card { background:#101720 !important; border-color:#1f2937 !important; }
    .m-head { color:#ffffff !important; }
    .m-txt  { color:#cbd5e1 !important; }
    .m-table-hd { background:#0e141d !important; }
    .m-th { color:#9fb3c8 !important; border-bottom:1px solid #233143 !important; }
    .m-td { color:#e5e7eb !important; border-bottom:1px solid #233143 !important; }
    .m-li { color:#e5e7eb !important; }
    .m-btn-pri { background:#3b82f6 !important; color:#fff !important; }
    .m-btn-sec { background:#1f2937 !important; color:#e5e7eb !important; }
    .m-pill { background:#2b3443 !important; color:#e5e7eb !important; }
    a { color:#93c5fd !important; }
  }
</style>
</head>
<body class="m-wrap" style="margin:0;padding:0;background:#f1f5f9;color:#0f172a;font:14px/1.4 system-ui, -apple-system, Segoe UI, Roboto, Arial, sans-serif;">
  <div style="height:24px;"></div>
  <div class="m-card" style="max-width:680px;margin:0 auto;background:#ffffff;border:1px solid #e5e7eb;border-radius:12px;overflow:hidden">
    <div style="padding:20px 20px 0 20px">
      <h1 class="m-head" style="margin:0 0 8px 0;font:800 20px/1.2 system-ui;">Sitio recuperado · ' . esc_html( $host ) . '</h1>
      <p class="m-txt" style="margin:0 0 16px 0;color:#475569">El monitor ha detectado que el sitio vuelve a estar en línea. No se han detectado incidencias en la última comprobación.</p>
    </div>

    <div class="m-table-hd" style="background:#f8fafc;padding:16px 20px;">
      <ul style="margin:0;padding-left:18px;color:inherit;font:400 14px/1.6 system-ui;">' . $li . '</ul>

      <table style="width:100%;border-collapse:collapse;margin-top:10px;">
        <thead>
          <tr>
            <th class="m-th" style="text-align:left;padding:10px;border-bottom:1px solid #e5e7eb;color:#334155;font:600 12px/1 system-ui;">Campo</th>
            <th class="m-th" style="text-align:left;padding:10px;border-bottom:1px solid #e5e7eb;color:#334155;font:600 12px/1 system-ui;">Valor</th>
          </tr>
        </thead>
        <tbody>
          <tr><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#334155;font:600 12px/1 system-ui;">Estado</td><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#0f172a;font:400 13px/1.4 system-ui;">' . $statusBadge . '</td></tr>
          <tr><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#334155;font:600 12px/1 system-ui;">HTTP</td><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#0f172a;font:400 13px/1.4 system-ui;">' . esc_html( $http ) . '</td></tr>
          <tr><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#334155;font:600 12px/1 system-ui;">Último chequeo</td><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#0f172a;font:400 13px/1.4 system-ui;">' . esc_html( $checked ) . '</td></tr>
          <tr><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#334155;font:600 12px/1 system-ui;">SSL caduca</td><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#0f172a;font:400 13px/1.4 system-ui;">' . esc_html( $sslDate ) . '</td></tr>
          <tr><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#334155;font:600 12px/1 system-ui;">Días SSL</td><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#0f172a;font:400 13px/1.4 system-ui;">' . $daysPill . '</td></tr>
          <tr><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#334155;font:600 12px/1 system-ui;">URL</td><td class="m-td" style="padding:10px;border-bottom:1px solid #e5e7eb;color:#0f172a;font:400 13px/1.4 system-ui;"><a href="' . $siteUrl . '" style="color:#2563eb;text-decoration:none;">' . $siteUrl . '</a></td></tr>
        </tbody>
      </table>

      <div style="margin-top:16px;display:flex;gap:10px;">
        <a class="m-btn-pri" href="' . $siteUrl . '" style="display:inline-block;background:#2563eb;color:#fff;text-decoration:none;padding:10px 14px;border-radius:8px;font:600 14px/1 system-ui;" target="_blank" rel="noreferrer">Abrir sitio</a>
        <a class="m-btn-sec" href="' . $adminUrl . '" style="display:inline-block;background:#0f172a;color:#fff;text-decoration:none;padding:10px 14px;border-radius:8px;font:600 14px/1 system-ui;">Ver en Site Monitor</a>
      </div>
    </div>

    <div style="padding:16px 20px;background:#ffffff;color:#64748b;font:400 12px/1.6 system-ui;">
      Este correo se envía a ' . $this->esc_for_email( $admin_to ) . '.<br>
      Umbral SSL: ' . $days_before . ' días · Reenvío cada ' . $throttle_hours . ' h.
    </div>
  </div>
  <div style="height:24px;"></div>
</body>
</html>';

  return [ $subject, $html ];
}



  public function apply_smtp( $phpmailer ) {
    $settings = get_option( $this->option_settings, [] );
    if ( empty( $settings['smtp_enabled'] ) ) return;

    // Base
    $host       = trim( (string) ( $settings['smtp_host'] ?? '' ) );
    $port       = (int)  ( $settings['smtp_port'] ?? 587 );
    $secure     = (string)( $settings['smtp_secure'] ?? '' ); // '', 'tls', 'ssl'
    $auth_flag  = !empty( $settings['smtp_auth'] );
    $username   = trim( (string) ( $settings['smtp_username'] ?? '' ) );
    $password   = (string)( $settings['smtp_password'] ?? '' );
    $from_email = trim( (string) ( $settings['smtp_from_email'] ?? '' ) );
    $from_name  = (string)( $settings['smtp_from_name'] ?? 'LMN Site Monitor' );

    // Forzar SMTP
    $phpmailer->isSMTP();
    $phpmailer->Host       = $host;
    $phpmailer->Port       = $port ?: 587;

    // Si hay usuario/clave, fuerza auth
    if ( !$auth_flag && $username && $password ) {
      $auth_flag = true;
    }
    $phpmailer->SMTPAuth   = $auth_flag;

    // Gmail/Google
    $is_gmail = stripos($host, 'smtp.gmail.com') !== false || stripos($host, 'googlemail.com') !== false;
    if ( $is_gmail ) {
      $phpmailer->SMTPSecure  = 'tls';
      $phpmailer->Port        = 587;
      $phpmailer->SMTPAuth    = true;
      $phpmailer->SMTPAutoTLS = true;
    } else {
      $phpmailer->SMTPSecure  = $secure ?: '';
      $phpmailer->SMTPAutoTLS = true;
    }

    if ( $phpmailer->SMTPAuth ) {
      if ( $username ) $phpmailer->Username = $username;
      if ( $password ) $phpmailer->Password = $password;
    }

    // Remitente / envelope
    $visible_from_email = $from_email ?: $username;
    if ( $is_gmail ) {
      $effective_from = $username ?: $visible_from_email;
      if ( $effective_from ) {
        $phpmailer->setFrom( $effective_from, $from_name, false );
        $phpmailer->Sender = $effective_from;
      }
      if ( $from_email && $username && strcasecmp($from_email, $username) !== 0 ) {
        $phpmailer->clearReplyTos();
        $phpmailer->addReplyTo( $from_email, $from_name );
      }
    } else {
      $effective_from = $visible_from_email;
      if ( $effective_from ) {
        $phpmailer->setFrom( $effective_from, $from_name, false );
        $phpmailer->Sender = $effective_from;
      }
    }

    // TLS/SSL options
    $phpmailer->SMTPOptions = [
      'ssl' => [
        'verify_peer'       => true,
        'verify_peer_name'  => true,
        'allow_self_signed' => false,
      ],
    ];

    $phpmailer->Timeout = 20;
    $phpmailer->SMTPDebug = 0;
  }
  public function handle_wp_mail_failed( $wp_error ) {
    $msg = 'LMN Site Monitor: el envío de email ha fallado. ' . ( $wp_error instanceof WP_Error ? $wp_error->get_error_message() : (string)$wp_error );
    // TTL 180 segundos
    set_transient( 'lmn_smtp_error', $msg, 180 );
  }
  public function handle_wp_mail_succeeded( $mail_data ) {
    // Si el envío ha sido correcto, borra el error previo
    delete_transient( 'lmn_smtp_error' );
  }



public function on_mail_failed( $wp_error ) {
  set_transient( 'lmn_mail_recent_failure', $wp_error->get_error_message(), 60 );
}
public function admin_notice_mail_failed() {
  if ( !current_user_can( 'manage_options' ) ) return;
  $msg = get_transient( 'lmn_mail_recent_failure' );
  if ( !$msg ) return;
  $url = admin_url( 'admin.php?page=' . self::MENU_SLUG . '&tab=config' );
  echo '<div class="notice notice-error"><p><strong>LMN Site Monitor:</strong> el envío de email ha fallado. '
    . esc_html( $msg ) . ' &nbsp; <a href="' . esc_url( $url ) . '">Configura SMTP</a>.</p></div>';
}

}

new LM_Site_Monitor_Classic();

// Añadir enlaces personalizados en la lista de plugins
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), function ( $links ) {
  // Enlace a la página de configuración del plugin
  $settings_link = '<a href="admin.php?page=lmn-site-monitor">' . __( 'Configuración', 'lmn-site-monitor' ) . '</a>'; //

 // Enlace a la donación por PayPal
$donate_url = 'https://www.paypal.com/donate/?business=FTTYSUMRSPWKN&no_recurring=0&currency_code=EUR&item_name=Cada+donativo+marca+la+diferencia.+Tu+aportaci%C3%B3n+me+permitir%C3%A1+seguir+mejorando+LMN+Site+Monitor.+%C2%A1Muchas+Gracias%21&amount=5.00';
$donate_link = '<a href="' . esc_url( $donate_url ) . '" target="_blank" rel="noopener noreferrer" style="color:#0073aa;font-weight:600;">' . __( 'Hacer una donación', 'lmn-site-monitor' ) . '</a>';
    // Insertamos los nuevos enlaces antes del enlace "Desactivar"
  array_unshift( $links, $settings_link, $donate_link );

  return $links;
});
